--------------------------------------------------------------------------------
-- Module: OV7670 Simple DMA - Version simplifiée
-- Description: Capture 640x480 et écrit en continu à l'adresse 0x30000000
-- Target: DE0-Nano-SoC (Cyclone V)
--------------------------------------------------------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

entity ov7670_simple_dma is
    port (
        -- Horloge et reset
        clk              : in  std_logic;  -- 50 MHz système
        reset_n          : in  std_logic;
        
        -- Interface OV7670
        cam_pclk         : in  std_logic;  -- Pixel clock de la caméra
        cam_href         : in  std_logic;  -- Horizontal reference
        cam_vsync        : in  std_logic;  -- Vertical sync
        cam_data         : in  std_logic_vector(7 downto 0);  -- Données pixel
        
        -- Avalon-MM Master (pour DMA vers HPS)
        avm_address      : out std_logic_vector(31 downto 0);
        avm_write        : out std_logic;
        avm_writedata    : out std_logic_vector(31 downto 0);
        avm_byteenable   : out std_logic_vector(3 downto 0);
        avm_waitrequest  : in  std_logic
    );
end entity ov7670_simple_dma;

architecture rtl of ov7670_simple_dma is
    
    -- Constantes
    constant BASE_ADDRESS  : unsigned(31 downto 0) := x"30000000";
    constant IMAGE_WIDTH   : integer := 640;
    constant IMAGE_HEIGHT  : integer := 480;
    constant TOTAL_PIXELS  : integer := IMAGE_WIDTH * IMAGE_HEIGHT;
    
    -- FIFO simple pour stocker les pixels RGB565
    type fifo_array is array (0 to 255) of std_logic_vector(15 downto 0);
    signal pixel_fifo      : fifo_array;
    signal fifo_wr_ptr     : unsigned(7 downto 0) := (others => '0');
    signal fifo_rd_ptr     : unsigned(7 downto 0) := (others => '0');
    signal fifo_count      : unsigned(8 downto 0) := (others => '0');
    
    -- Capture de la caméra
    signal pclk_d          : std_logic_vector(2 downto 0) := (others => '0');
    signal href_d          : std_logic := '0';
    signal vsync_d         : std_logic_vector(1 downto 0) := (others => '0');
    signal byte_count      : std_logic := '0';
    signal pixel_byte_high : std_logic_vector(7 downto 0);
    signal pixel_counter   : unsigned(18 downto 0) := (others => '0');
    
    -- DMA
    signal dma_address     : unsigned(31 downto 0) := BASE_ADDRESS;
    signal dma_active      : std_logic := '0';
    
    -- Détection fronts
    signal pclk_rising     : std_logic;
    signal vsync_rising    : std_logic;
    
begin

    -- Détection des fronts
    pclk_rising  <= '1' when (pclk_d(2 downto 1) = "01") else '0';
    vsync_rising <= '1' when (vsync_d = "01") else '0';
    
    ----------------------------------------------------------------------------
    -- Synchronisation des signaux de la caméra
    ----------------------------------------------------------------------------
    process(clk)
    begin
        if rising_edge(clk) then
            pclk_d  <= pclk_d(1 downto 0) & cam_pclk;
            href_d  <= cam_href;
            vsync_d <= vsync_d(0) & cam_vsync;
        end if;
    end process;
    
    ----------------------------------------------------------------------------
    -- Capture des pixels (RGB565 = 2 bytes par pixel)
    ----------------------------------------------------------------------------
    process(clk)
    begin
        if rising_edge(clk) then
            if reset_n = '0' then
                byte_count <= '0';
                pixel_counter <= (others => '0');
                fifo_wr_ptr <= (others => '0');
                
            else
                -- Reset au début d'une nouvelle frame
                if vsync_rising = '1' then
                    pixel_counter <= (others => '0');
                    byte_count <= '0';
                end if;
                
                -- Capture sur front montant de PCLK et pendant HREF
                if pclk_rising = '1' and href_d = '1' then
                    if byte_count = '0' then
                        -- Premier byte (RGB565 MSB)
                        pixel_byte_high <= cam_data;
                        byte_count <= '1';
                    else
                        -- Deuxième byte (RGB565 LSB) - pixel complet
                        if pixel_counter < TOTAL_PIXELS and fifo_count < 256 then
                            pixel_fifo(to_integer(fifo_wr_ptr)) <= pixel_byte_high & cam_data;
                            fifo_wr_ptr <= fifo_wr_ptr + 1;
                            pixel_counter <= pixel_counter + 1;
                        end if;
                        byte_count <= '0';
                    end if;
                end if;
            end if;
        end if;
    end process;
    
    ----------------------------------------------------------------------------
    -- Gestion du compteur FIFO
    ----------------------------------------------------------------------------
    process(clk)
    begin
        if rising_edge(clk) then
            if reset_n = '0' then
                fifo_count <= (others => '0');
            else
                -- Écriture et lecture en même temps
                if (pclk_rising = '1' and href_d = '1' and byte_count = '1' and 
                    pixel_counter < TOTAL_PIXELS and fifo_count < 256) and 
                   (dma_active = '1' and avm_waitrequest = '0') then
                    fifo_count <= fifo_count;  -- Pas de changement
                    
                -- Seulement écriture dans FIFO
                elsif (pclk_rising = '1' and href_d = '1' and byte_count = '1' and 
                       pixel_counter < TOTAL_PIXELS and fifo_count < 256) then
                    fifo_count <= fifo_count + 2;  -- +2 car on écrit 2 pixels à la fois
                    
                -- Seulement lecture de FIFO (DMA)
                elsif (dma_active = '1' and avm_waitrequest = '0') then
                    if fifo_count >= 2 then
                        fifo_count <= fifo_count - 2;
                        fifo_rd_ptr <= fifo_rd_ptr + 2;
                    end if;
                end if;
            end if;
        end if;
    end process;
    
    ----------------------------------------------------------------------------
    -- Contrôleur DMA simplifié
    ----------------------------------------------------------------------------
    process(clk)
    begin
        if rising_edge(clk) then
            if reset_n = '0' then
                dma_address <= BASE_ADDRESS;
                dma_active <= '0';
                avm_write <= '0';
                
            else
                -- Reset de l'adresse au début de chaque frame
                if vsync_rising = '1' then
                    dma_address <= BASE_ADDRESS;
                end if;
                
                -- Transfert DMA si au moins 2 pixels disponibles dans FIFO
                if fifo_count >= 2 then
                    dma_active <= '1';
                    avm_write <= '1';
                    
                    -- Si le transfert est accepté (waitrequest = 0)
                    if avm_waitrequest = '0' then
                        dma_address <= dma_address + 4;  -- +4 bytes (32 bits)
                    end if;
                else
                    dma_active <= '0';
                    avm_write <= '0';
                end if;
            end if;
        end if;
    end process;
    
    -- Assignation des signaux Avalon
    avm_address    <= std_logic_vector(dma_address);
    avm_byteenable <= "1111";  -- Tous les bytes valides
    
    -- Combiner 2 pixels RGB565 (16 bits chacun) en un mot de 32 bits
    avm_writedata  <= pixel_fifo(to_integer(fifo_rd_ptr)) & 
                      pixel_fifo(to_integer(fifo_rd_ptr + 1));
    
end architecture rtl;
