# OV7670 Simple DMA - Version Simplifiée

## Vue d'ensemble

Version ultra-simplifiée qui écrit en continu à l'adresse **0x30000000** de la DDR.

## Simplifications apportées

✅ **Pas de registres de configuration** - Adresse fixe en dur  
✅ **Pas d'interface Avalon Slave** - Seulement Master pour DMA  
✅ **Pas d'interruption** - Écriture continue  
✅ **Pas de burst** - Transferts mot par mot  
✅ **FIFO réduite** - 256 pixels au lieu de 512  
✅ **Pas de compteur de frames** - Écrasement automatique  

## Fonctionnement

1. **Capture** : La caméra envoie les pixels en RGB565
2. **FIFO** : Les pixels sont stockés temporairement 
3. **DMA** : Dès que 2 pixels sont disponibles, ils sont écrits en DDR
4. **Boucle** : À chaque nouvelle frame (VSYNC), l'adresse repart à 0x30000000

```
Frame 1 → 0x30000000 - 0x30096000
Frame 2 → 0x30000000 - 0x30096000  (écrase Frame 1)
Frame 3 → 0x30000000 - 0x30096000  (écrase Frame 2)
...
```

## Intégration Qsys

```tcl
# ov7670_simple_dma_hw.tcl

package require -exact qsys 16.0

set_module_property NAME ov7670_simple_dma
set_module_property VERSION 1.0
set_module_property DESCRIPTION "OV7670 Simple DMA to 0x30000000"

# Fichiers sources
add_fileset synth_fileset QUARTUS_SYNTH synth_callback
set_fileset_property synth_fileset TOP_LEVEL ov7670_simple_dma

proc synth_callback { entity_name } {
    add_fileset_file ov7670_simple_dma.vhd VHDL PATH ov7670_simple_dma.vhd TOP_LEVEL_FILE
}

# Clock et Reset
add_interface clock clock end
add_interface_port clock clk clk Input 1
add_interface reset reset end
add_interface_port reset reset_n reset_n Input 1

# Avalon-MM Master (DMA uniquement)
add_interface avalon_master avalon start
set_interface_property avalon_master addressUnits SYMBOLS
set_interface_property avalon_master burstOnBurstBoundariesOnly false
set_interface_property avalon_master doStreamReads false
set_interface_property avalon_master doStreamWrites false

add_interface_port avalon_master avm_address address Output 32
add_interface_port avalon_master avm_write write Output 1
add_interface_port avalon_master avm_writedata writedata Output 32
add_interface_port avalon_master avm_byteenable byteenable Output 4
add_interface_port avalon_master avm_waitrequest waitrequest Input 1

# Signaux caméra (conduit)
add_interface camera conduit end
add_interface_port camera cam_pclk export Input 1
add_interface_port camera cam_href export Input 1
add_interface_port camera cam_vsync export Input 1
add_interface_port camera cam_data export Input 8
```

## Schéma de connexion Qsys

```
┌──────────────────┐
│  OV7670 Camera   │
│  (Hardware)      │
└────────┬─────────┘
         │ PCLK, HREF, VSYNC, DATA[7:0]
         │
┌────────▼─────────────┐
│ ov7670_simple_dma    │
│                      │
│  - Capture pixels    │
│  - FIFO 256 pixels   │
│  - DMA continu       │
└────────┬─────────────┘
         │ Avalon-MM Master
         │
┌────────▼─────────────┐
│  HPS DDR Controller  │
│                      │
│  0x30000000          │
│  640x480x2 = 614KB   │
└──────────────────────┘
```

## Réservation mémoire Linux

Pour que Linux ne touche pas à la zone 0x30000000, modifiez le Device Tree :

```dts
/ {
    reserved-memory {
        #address-cells = <1>;
        #size-cells = <1>;
        ranges;

        camera_buffer: camera@30000000 {
            compatible = "shared-dma-pool";
            reg = <0x30000000 0x00100000>;  /* 1 MB réservé */
            no-map;
        };
    };
};
```

Ou via bootargs :
```bash
setenv bootargs 'mem=768M'  # Linux s'arrête à 768M, laisse le reste libre
```

## Lecture de l'image depuis Linux

### Code C pour lire l'image

```c
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <unistd.h>
#include <stdint.h>

#define CAM_BASE_ADDR 0x30000000
#define IMAGE_WIDTH   640
#define IMAGE_HEIGHT  480
#define IMAGE_SIZE    (IMAGE_WIDTH * IMAGE_HEIGHT * 2)

void save_raw_image(const char *filename, uint16_t *image)
{
    FILE *fp = fopen(filename, "wb");
    if (!fp) {
        perror("fopen");
        return;
    }
    
    fwrite(image, 1, IMAGE_SIZE, fp);
    fclose(fp);
    printf("Image sauvegardée: %s\n", filename);
}

void save_ppm(const char *filename, uint16_t *image)
{
    FILE *fp = fopen(filename, "wb");
    if (!fp) {
        perror("fopen");
        return;
    }
    
    fprintf(fp, "P6\n%d %d\n255\n", IMAGE_WIDTH, IMAGE_HEIGHT);
    
    for (int i = 0; i < IMAGE_WIDTH * IMAGE_HEIGHT; i++) {
        uint16_t rgb565 = image[i];
        
        // Conversion RGB565 → RGB888
        uint8_t r = ((rgb565 >> 11) & 0x1F) * 255 / 31;
        uint8_t g = ((rgb565 >> 5) & 0x3F) * 255 / 63;
        uint8_t b = (rgb565 & 0x1F) * 255 / 31;
        
        fwrite(&r, 1, 1, fp);
        fwrite(&g, 1, 1, fp);
        fwrite(&b, 1, 1, fp);
    }
    
    fclose(fp);
    printf("Image PPM sauvegardée: %s\n", filename);
}

int main(int argc, char *argv[])
{
    int mem_fd;
    void *mapped_base;
    uint16_t *image;
    
    // Ouvrir /dev/mem
    mem_fd = open("/dev/mem", O_RDONLY | O_SYNC);
    if (mem_fd < 0) {
        perror("Impossible d'ouvrir /dev/mem");
        return 1;
    }
    
    // Mapper la région mémoire de la caméra
    mapped_base = mmap(NULL, IMAGE_SIZE, PROT_READ, 
                      MAP_SHARED, mem_fd, CAM_BASE_ADDR);
    
    if (mapped_base == MAP_FAILED) {
        perror("mmap");
        close(mem_fd);
        return 1;
    }
    
    image = (uint16_t *)mapped_base;
    
    printf("Capture de l'image à 0x%08X...\n", CAM_BASE_ADDR);
    
    // Attendre un peu pour être sûr qu'une frame soit capturée
    sleep(1);
    
    // Sauvegarder l'image
    save_raw_image("capture.raw", image);
    save_ppm("capture.ppm", image);
    
    printf("Terminé!\n");
    
    // Nettoyer
    munmap(mapped_base, IMAGE_SIZE);
    close(mem_fd);
    
    return 0;
}
```

### Compilation et exécution

```bash
# Sur le PC de développement (cross-compilation)
arm-linux-gnueabihf-gcc -o capture_image capture_image.c

# Copier sur la DE0-Nano-SoC
scp capture_image root@192.168.1.100:/root/

# Sur la DE0-Nano-SoC
chmod +x capture_image
./capture_image

# Récupérer l'image
scp root@192.168.1.100:/root/capture.ppm .
```

## Script Python pour visualiser

```python
#!/usr/bin/env python3
import numpy as np
import matplotlib.pyplot as plt
from PIL import Image

def read_rgb565(filename, width=640, height=480):
    """Lit un fichier RAW RGB565 et le convertit en RGB888"""
    with open(filename, 'rb') as f:
        data = np.fromfile(f, dtype=np.uint16, count=width*height)
    
    # Extraire les composantes RGB du format RGB565
    r = ((data >> 11) & 0x1F) * 255 // 31
    g = ((data >> 5) & 0x3F) * 255 // 63
    b = (data & 0x1F) * 255 // 31
    
    # Créer l'image RGB
    img = np.zeros((height, width, 3), dtype=np.uint8)
    img[:,:,0] = r.reshape(height, width)
    img[:,:,1] = g.reshape(height, width)
    img[:,:,2] = b.reshape(height, width)
    
    return img

# Lire et afficher
img = read_rgb565('capture.raw')
plt.imshow(img)
plt.title('Capture OV7670')
plt.axis('off')
plt.show()

# Sauvegarder en PNG
Image.fromarray(img).save('capture.png')
print("Image sauvegardée: capture.png")
```

## Performances

### Calculs théoriques

- **Taille image** : 640 × 480 × 2 = 614 400 bytes
- **Framerate OV7670** : 30 fps max
- **Bande passante** : 614 KB × 30 = 18.4 MB/s
- **Temps par frame** : 33.3 ms

### Timing DMA

Avec un bus Avalon à 100 MHz :
- 1 transfert = 1 cycle (sans waitrequest)
- 614400 bytes / 4 bytes = 153600 transferts
- Temps minimum = 153600 cycles = 1.536 ms
- **Largement suffisant pour 30 fps !**

## Avantages/Inconvénients

### ✅ Avantages
- Code VHDL très simple (~150 lignes vs 400+)
- Pas de registres à configurer
- Toujours actif dès le reset
- Facile à déboguer

### ❌ Inconvénients
- Adresse hardcodée (pas flexible)
- Pas de contrôle start/stop
- Pas d'interruption (polling nécessaire)
- Écrase toujours la même zone mémoire

## Déboggage

### Vérifier que ça fonctionne

```bash
# Lire les premiers bytes à 0x30000000
devmem2 0x30000000
devmem2 0x30000004
devmem2 0x30000008

# Si les valeurs changent = caméra fonctionne!
```

### SignalTap (pour debug FPGA)

Signaux à observer :
- `cam_vsync` : Doit pulser à ~30 Hz
- `cam_href` : Doit être haut pendant les lignes actives
- `cam_pclk` : Clock pixel
- `fifo_count` : Doit varier entre 0 et ~256
- `dma_address` : Doit incrémenter de 0x30000000 à 0x30096000

## Configuration I2C de la caméra

N'oubliez pas de configurer la caméra OV7670 via I2C avant utilisation !

```c
// Registres essentiels
OV7670_write(0x12, 0x80);  // Reset
delay_ms(10);
OV7670_write(0x12, 0x00);  // RGB mode
OV7670_write(0x40, 0xD0);  // RGB565 output
OV7670_write(0x8C, 0x00);  // RGB565
OV7670_write(0x04, 0x00);  // COM1
OV7670_write(0x15, 0x20);  // COM10 - VSYNC negative
// ... voir datasheet pour config complète
```

## Checklist d'intégration

- [ ] VHDL compilé dans Quartus
- [ ] Composant ajouté dans Qsys
- [ ] Connexion Avalon Master → HPS DDR
- [ ] Pins caméra assignées dans .qsf
- [ ] Mémoire 0x30000000 réservée dans Linux
- [ ] Caméra configurée via I2C
- [ ] XCLK 24MHz généré pour la caméra
- [ ] Test de lecture mémoire fonctionnel

Voilà ! C'est beaucoup plus simple maintenant 🎥
